<?php

namespace App\Http\Controllers;

use App\Models\Exercise;
use App\Models\Category;
use App\Services\VimeoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;
use Log;
use RealRashid\SweetAlert\Facades\Alert;
use Str;

class ExerciseController extends Controller
{

    protected $vimeo;
    public function __construct(VimeoService $vimeoService) {
        $this->vimeo = $vimeoService;
    }

    public function index(Request $request)
    {

        $page = $request->page;

        if(isset($page) && is_numeric($page) && $page>0) {

            $records_to_show = 25;
            $records_to_skip = ($page - 1) * $records_to_show;

            $query = Exercise::query();

            $all_records_count = $query->count();

            $num_of_pages = (int) ceil($all_records_count / $records_to_show);
            
            $exercises = $query->skip($records_to_skip)
            ->take($records_to_show)
            ->get();

            return view('exercises.index', compact('exercises','num_of_pages', 'page'));

            
        } else {
            return redirect()->route('exercises.index', ['page'=>1]);
        }

    }

    public function create() {

        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $categories = Category::where('deleted', 0)->select('id', 'name')->get();

        return view('exercises.form', compact('categories'));
    }

    public function store(Request $request) {
        $insert = 0;
        $error = '';

        try {

            $this->requestValidations();

            $inputs = $request->all();

            // Upload Main Image
            $imgName = $this->uploadFile($request, 'main_image', 'exercises_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:2048');
            $inputs['main_image'] = $imgName;

            // Store
            $insert = Exercise::create($inputs);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($insert) {
            session()->flash('success' , 'Record Created Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
    }

    public function show($id) {
        
        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $exercise = Exercise::findOrFail($id);
        $categories = Category::where('deleted', 0)->select('id', 'name')->get();

        return view('exercises.form', compact('exercise', 'categories'));
    }

    public function update(Request $request, Exercise $exercise, $id)
    {

        $status = 0;
        $error = '';

        try {

            $this->requestValidations($id);
            $exercise = Exercise::findOrFail($id);

            $inputs = $request->all();

            if($request->main_image != null) {
                // Upload Image
                $imgName = $this->uploadFile($request, 'main_image', 'exercises_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:2048');
                $inputs['main_image'] = $imgName;
            }

            $exercise->fill($inputs);

            if($exercise->save()) {
                $status = 1;
            }
            
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($status) {
            session()->flash('success' , 'Record Updated Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
        
    }

    public function destroy(Request $request)
    {
        $status = 0;
        $message = 'Something went wrong !';

        $delete = Exercise::where('id', $request->record_id)->delete();

        if($delete) {

            $status = 1;
            $message = "Record Deleted Successfully !";
        }

        return response()->json([
            'status'=>$status,
            'message'=>$message
        ]);
    }

    public function requestValidations($record_id = null) {

        $slugUniqueRule = Rule::unique('exercises', 'slug')->ignore($record_id);

        return request()->validate([
            'title' => ['required', 'string', 'max:256'],
            'overview' => ['nullable'],
            'description' => ['nullable'],
            'duration' => ['nullable', 'integer', 'min:1', 'max:7200'], 
            'sets' => ['nullable', 'integer', 'min:1', 'max:1000'], 
            'difficulty' => ['required', 'in:Beginner,Intermediate,Advanced'],
        ]);
    }

    public function uploadFile(Request $request, $file_name, $folder_name, $rules) {

        // Get the uploaded file
        $file = $request->file($file_name);
    
        // Validation rules for general files
        $fileRules = [
            $file_name => $rules,
        ];
    
        // Validate the file input
        $this->validate($request, $fileRules);
    
        // Generate a unique name for the file
        $fileName = uniqid().'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
    
        // Specify the directory where you want to save the file
        $directory = 'uploads/'.$folder_name;
    
        // Create the directory if it doesn't exist
        File::makeDirectory($directory, $mode = 0777, true, true);
    
        // Move the uploaded file to the specified directory
        $file->move($directory, $fileName);
    
        // Return the path of the uploaded file
        return $folder_name.'/'.$fileName;
    }

    public function uploadVimeoVideo(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'video' => 'required|mimes:mp4,mov,avi|max:50000',
            'title' => 'required',
            'description' => 'nullable'
        ]);

        if(!$validator->fails()) {
            $filePath = $request->file('video')->getPathname();
            $title = $request->title;
            $description = $request->description;

            $exercise = Exercise::find($request->record_id);

            if($exercise) {
                $response = $this->vimeo->uploadVideo($filePath, $title, $description);
                $videoId = str_replace("/videos/", "", $response);

                if($videoId) {

                    $getVideo =  $this->vimeo->getVideo($videoId);

                    $player_embed_url = isset($getVideo['body']['player_embed_url']) ? $getVideo['body']['player_embed_url'] : '';

                    $exercise->fill(["video"=>$videoId, "player_embed_url" => $player_embed_url]);
                    if($exercise->save()) {
                        session()->flash('success' , 'Video Uplaoded Successfully.');
                    } else {
                        session()->flash('error' , 'Failed to upload video.');
                    }
                } else {
                    session()->flash('error' , 'Failed to upload to vimue.');
                }
            } else {
                session()->flash('error' , 'Record Not Found.');
            }

        } else {
            session()->flash('error' , $validator->errors()->first());
        }
 
        return redirect()->back();
    }
    

}
