<?php

namespace App\Http\Controllers;

use App\Models\Activity;
use App\Models\ActivityToExercise;
use App\Models\Category;
use App\Models\Type;
use App\Models\Exercise;
use App\Services\VimeoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;
use Log;
use RealRashid\SweetAlert\Facades\Alert;
use Str;

class ActivityController extends Controller
{

    protected $vimeo;
    public function __construct(VimeoService $vimeoService) {
        $this->vimeo = $vimeoService;
    }

    public function index(Request $request)
    {

        $page = $request->page;

        if(isset($page) && is_numeric($page) && $page>0) {

            $records_to_show = 25;
            $records_to_skip = ($page - 1) * $records_to_show;

            $query = Activity::query();

            $all_records_count = $query->count();

            $num_of_pages = (int) ceil($all_records_count / $records_to_show);
            
            $activities = $query->skip($records_to_skip)
            ->take($records_to_show)
            ->with('category')
            ->get();

            return view('activities.index', compact('activities','num_of_pages', 'page'));

            
        } else {
            return redirect()->route('activities.index', ['page'=>1]);
        }

    }

    public function create() {

        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $categories = Category::where('deleted', 0)->select('id', 'name')->get();
        $types = Type::all();

        return view('activities.form', compact('categories', 'types'));
    }

    public function store(Request $request) {
        $insert = 0;
        $error = '';

        try {

            $this->requestValidations();

            $inputs = $request->all();

            // Upload Main Image
            $imgName = $this->uploadFile($request, 'main_image', 'activities_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:1024');
            $inputs['main_image'] = $imgName;

            // Store
            $insert = Activity::create($inputs);

        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($insert) {
            session()->flash('success' , 'Record Created Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
    }

    public function show($id) {
        
        if(session()->has('success')) {
            Alert::success('success', session()->get('success'));
        }

        if(session()->has('error')) {
            Alert::error('error', session()->get('error'));
        }

        $activity = Activity::findOrFail($id);
        $categories = Category::where('deleted', 0)->select('id', 'name')->get();
        $types = Type::all();
        $exercises =  Exercise::all();
        $assigned_exercises =  ActivityToExercise::where('activity_id', $id)->with("exercise")->get();

        return view('activities.form', compact('activity', 'categories', 'types', 'exercises', 'assigned_exercises'));
    }

    public function update(Request $request, Activity $activity, $id)
    {

        $status = 0;
        $error = '';

        try {

            $this->requestValidations($id);
            $activity = Activity::findOrFail($id);

            $inputs = $request->all();

            if($request->main_image != null) {
                // Upload Image
                $imgName = $this->uploadFile($request, 'main_image', 'activities_imgs', 'required|mimes:jpeg,jpg,png,webp,avif|max:1024');
                $inputs['main_image'] = $imgName;
            }

            $activity->fill($inputs);

            if($activity->save()) {
                $status = 1;
            }
            
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($status) {
            session()->flash('success' , 'Record Updated Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , 'Something went wrong.');
        }
       
        return redirect()->back();
        
    }

    public function destroy(Request $request)
    {
        $status = 0;
        $message = 'Something went wrong !';

        $delete = Activity::where('id', $request->record_id)->delete();

        if($delete) {

            $status = 1;
            $message = "Record Deleted Successfully !";
        }

        return response()->json([
            'status'=>$status,
            'message'=>$message
        ]);
    }
    
    public function assign_activity_exercise(Request $request) {
        $status = 0;
        $error = 'Something went wrong.';

        try {
            
            $is_exists =  ActivityToExercise::where('activity_id', $request->activity_id)->where('exercise_id', $request->exercise_id)->exists();
            
            if(!$is_exists) {
                $insert = ActivityToExercise::create([
                   "activity_id" => $request->activity_id, 
                   "exercise_id" => $request->exercise_id, 
                   "sets" => $request->sets, 
                   "reps" => $request->reps, 
                   "rest_between_sets" => $request->rest_between_sets, 
                   "rest_between_exercises" => $request->rest_between_exercises, 
                ]);
                
                if($insert) {
                    $status = 1;
                }
            } else {
                $error = 'Assigned already before.';
            }

            
            
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($status) {
            session()->flash('success' , 'Assigned Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , $error);
        }
       
        return redirect()->back();
    }
    
    public function delete_assigned_exercise(Request $request) {
        $status = 0;
        $error = 'Something went wrong.';

        try {
            
            $delete =  ActivityToExercise::where('id', $request->id)->delete();
            
            if($delete) {
                $status = 1;
            }
            
        } catch (ValidationException $e) {
            return redirect()->back()->withErrors($e->errors())->withInput();

        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if($status) {
            session()->flash('success' , 'Deleted Successfully !');
        } else {
            Log::error($error);
            session()->flash('error' , $error . $request->id);
        }
       
        return redirect()->back();
    }

    public function requestValidations($record_id = null) {

        $slugUniqueRule = Rule::unique('activities', 'slug')->ignore($record_id);

        return request()->validate([
            'title' => ['required', 'string', 'max:256'],
            'type_id' => ['required', 'numeric'],
            'category_id' => ['required', 'numeric'],
            'overview' => ['required'],
            'description' => ['required'],
            'duration' => ['required', 'integer', 'min:30', 'max:7200'], 
            'difficulty' => ['required', 'in:Beginner,Intermediate,Advanced'],
        ]);
    }

    public function uploadFile(Request $request, $file_name, $folder_name, $rules) {

        // Get the uploaded file
        $file = $request->file($file_name);
    
        // Validation rules for general files
        $fileRules = [
            $file_name => $rules,
        ];
    
        // Validate the file input
        $this->validate($request, $fileRules);
    
        // Generate a unique name for the file
        $fileName = uniqid().'_'.Str::random(6).'.'.$file->getClientOriginalExtension();
    
        // Specify the directory where you want to save the file
        $directory = 'uploads/'.$folder_name;
    
        // Create the directory if it doesn't exist
        File::makeDirectory($directory, $mode = 0777, true, true);
    
        // Move the uploaded file to the specified directory
        $file->move($directory, $fileName);
    
        // Return the path of the uploaded file
        return $folder_name.'/'.$fileName;
    }

    

}
